Imports System.Collections

'
' Instances of this class are responsible for managing the
' execution of commands.  More specifically, for the purposes
' of this word processing program, instances of this class are
' responsible for maintaining a command history for undo and
' redo.
'
Public Class CommandManager
    ' The single instance of this command.
    Private Shared myInstance As New CommandManager()

    ' The maximum number of command to keep in the history
    Private maxHistoryLength As Integer = 100

    Private history As New ArrayList()
    Private redoList As New ArrayList()

    '
    ' Returns the instance of this class.
    '
    Public Shared Function GetInstance() As CommandManager
        Return myInstance
    End Function 'GetInstance

    ' Private constructor to prevent other classes from
    ' instantiating this class.
    Private Sub New()
    End Sub 'New

    '
    ' Invoke a command and add it to the history. If the command
    ' object's doIt method was previously called, then it is
    ' expected to return false.
    '
    Public Sub InvokeCommand(ByVal command As ICommand)
        Dim cmdType As Type = CObj(command).GetType()
        Dim uType As Type = GetType(UndoAttribute)
        Dim undoAttr As Attribute
        undoAttr = Attribute.GetCustomAttribute(cmdType, uType)
        If undoAttr IsNot Nothing Then
            undo()
            Return
        End If
        Dim rType As Type = GetType(RedoAttribute)
        Dim redoAttr As Attribute
        redoAttr = Attribute.GetCustomAttribute(cmdType, rType)
        If redoAttr IsNot Nothing Then
            redo()
            Return
        End If
        If command.DoIt() Then
            ' doIt returned true, which means it can be undone
            addToHistory(command)
        Else ' command cannot be undone
            history.Clear()
        End If
        ' After a command that is not an undo or a redo, ensure that
        ' the redo list is empty.
        If redoList.Count > 0 Then
            redoList.Clear()
        End If
    End Sub 'InvokeCommand

    ' Undo the most recent command in the commmand history.
    Private Sub undo()
        If history.Count > 0 Then
            ' If there are commands in the history
            Dim undoCommand As ICommand
            undoCommand = CType(history(0), ICommand)
            history.RemoveAt(0)
            undoCommand.UndoIt()
            redoList.Insert(0, undoCommand)
        End If
    End Sub 'undo

    ' Redo the most recently undone command
    Private Sub redo()
        If redoList.Count > 0 Then
            ' If the redo list is not empty
            Dim redoCommand As ICommand
            redoCommand = CType(redoList(0), ICommand)
            redoList.RemoveAt(0)
            redoCommand.DoIt()
            history.Insert(0, redoCommand)
        End If
    End Sub 'redo

    ' Add a command to the command history.
    Private Sub addToHistory(ByVal command As ICommand)
        history.Insert(0, command)
        ' If size of history has exceded maxHistoryLength, remove
        ' the oldest command from the history
        If history.Count > maxHistoryLength Then
            history.RemoveAt((history.Count - 1))
        End If
    End Sub 'addToHistory
End Class 'CommandManager